close all;
clear all;
clc;
format long;

% Parameters
R_Leak = 10^8;              % [ohm]
C = 10^(-10);               % [F]
tau = R_Leak*C;             % [s]
E_Leak = -50e-3;            % [V]
V_th = 10e-3;               % [V]
V_reset = -70e-3;           % [V]

% Time and Current set
I_max = 2;                                              % [nA]
number_currents = 15;                                   % Number of points for current
I_values = linspace(0,I_max,number_currents)*1e-9;      % [A]

time_steps = 20000;                                     % Number of points for time
t_max = 200;                                            % [ms]
time = linspace(0,t_max,time_steps)*1e-3;               % [s]
t1_index = round(time_steps/5);
t1 = time(t1_index);                                    % [s]
t2_index = round(time_steps/1.2);
t2 = time(t2_index);                                    % [s]
t0 = 0;                                                 % [s]

% Time of injection
for t_i = 1 : time_steps
    injection_range(t_i) = heaviside(time(t_i)-t1)-...
                           heaviside(time(t_i)-t2);
end

% V_inf
for I_i = 1 : number_currents
    for t_i = 1 : time_steps
        V_inf(I_i,t_i) = E_Leak + ...
        injection_range(t_i)*I_values(I_i)*R_Leak;      % [V]
    end
end

% Spike generation in time
V_C_Spikes = zeros(number_currents,time_steps) ;

for I_i = 1 : number_currents
    
    Initial_cond = E_Leak;                              % [V]  
    Spike = 0;                                          % Flag
    Spikes_time_index(I_i,1) = 0;
    
    for t_i = 1 : time_steps
        
        if time(t_i) < t1                                               % Before the injection
            
            V_C_Spikes(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                                  V_inf(I_i,t_i))*exp(-(time(t_i)-...
                                  t0)/tau);                             % [V]
        
        elseif (time(t_i) >= t1) && (time(t_i) < t2)                    % During the injection
            
            if Spike == 0
                Initial_cond = V_C_Spikes(I_i,t1_index-1);              % [V]
                Initial_time = t1;
            else
                Initial_cond = V_reset;
                Initial_time = time(Spikes_time_index(I_i,end));
            end
            
            V_C_Spikes(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                                  V_inf(I_i,t_i))*exp(-(time(t_i)-...
                                  Initial_time)/tau);                   % [V]
        
        elseif time(t_i) >= t2                                          % After the injection
            
            Initial_cond = V_C_Spikes(I_i,t2_index-1);                  % [V]
            V_C_Spikes(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                                  V_inf(I_i,t_i))*exp(-(time(t_i)-...
                                  t2)/tau);                             % [V]
        end
        


        if V_C_Spikes(I_i,t_i) >= V_th
            V_C_Spikes(I_i,t_i) = V_reset;
            Spikes_time_index(I_i,end+1) = t_i;
            Spike = 1;                                                  % Flag, it means we had a spike
        end


    end

end



% Charging in time
V_C = zeros(number_currents,time_steps);

for I_i = 1 : number_currents
    
    Initial_cond = E_Leak;      % [V]
    
    for t_i = 1 : time_steps
        
        if time(t_i) < t1       % Before the injection
        
            V_C(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                           V_inf(I_i,t_i))*exp(-(time(t_i)-t0)/tau);      % [V]
        
        elseif (time(t_i) >= t1) && (time(t_i) < t2)       % During the injection
            
            Initial_cond = V_C(I_i,t1_index-1);                                     % [V]
            Initial_time = t1;
            
            V_C(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                           V_inf(I_i,t_i))*exp(-(time(t_i)-Initial_time)/tau);      % [V]
        
        elseif time(t_i) >= t2                              % After the injection
            
            Initial_cond = V_C(I_i,t2_index-1);             % [V]
            V_C(I_i,t_i) = V_inf(I_i,t_i) + (Initial_cond-...
                           V_inf(I_i,t_i))*exp(-(time(t_i)-t2)/tau);    % [V]
        
        end
    
    end

end


% Spike trains visualization in time
Spikes_generation = zeros(number_currents,time_steps);

for I_i = 1 : number_currents
    
    for t_i = 1 : time_steps
        
        if ismember(t_i,Spikes_time_index(I_i,:))
            Spikes_generation(I_i,t_i) = 1;
        end
    
    end

end


% Frequency vs Current curve
I_th = (R_Leak^-1*(V_th - E_Leak))*1e9;    % [nA]
I_max_fI = 2;                              % [nA]
number_fI_current = 140000;
I_values_fI = linspace(I_th,I_max_fI,number_fI_current)*1e-9;               % [A]

for I_i= 1: number_fI_current
    V_inf_fI = E_Leak + I_values_fI(I_i)*R_Leak;                            % [V]
    f_I_and_F(I_i) = (tau*log((V_inf_fI-V_reset)/(V_inf_fI-V_th)))^-1;      % [Hz]
end

linear_f = (I_values_fI - I_th*1e-9)/(C*(V_th-V_reset));                    % [Hz]


% Plot
E_Leak_vec = E_Leak*ones(1,time_steps);    % [V]
Vth_vec = V_th*ones(1,time_steps);         % [V]
Vreset_vec = V_reset*ones(1,time_steps);   % [V]


figure
subplot(2,1,1)
plot(time*1e3,V_C_Spikes(8,:)*1e3,'b');
hold on
plot(time*1e3,V_C(8,:)*1e3,'b--');
plot(time*1e3,E_Leak_vec*1e3,'m--');
plot(time*1e3,Vth_vec*1e3,'r--');
plot(time*1e3,Vreset_vec*1e3,'k--');
set(gca,'FontSize',20,'FontName','Times New Roman',...
    'XScale','lin','YScale','lin','box','on');
xlabel('Time, ms');
ylabel('V_{C}, mV');
grid on;
title('Integrate and fire model');
legend('V_{C}','Charging in time','E_{Leak}','V_{th}','V_{reset}');

subplot(2,1,2)
plot(time*1e3,Spikes_generation(8,:),'r');
set(gca,'FontSize',20,'FontName','Times New Roman',...
'XScale','lin','YScale','lin','box','on');
xlabel('Time, ms');
ylabel('Events');
grid on;
title('Spike train due to the injection with the I&F model');


figure
plot(I_values_fI*1e9,f_I_and_F,'b');
hold on
plot(I_values_fI*1e9,linear_f,'r');
set(gca,'FontSize',20,'FontName','Times New Roman',...
    'XScale','lin','YScale','lin','box','on');
xlabel('I_{Injected}, nA');
ylabel('Output frequency, Hz');
xlim([0,I_max_fI]);
grid on;
title('Frequency vs Current, Integrate and fire model');
legend('Real behaviour','Approximation for large I_{in}');










